/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/uniform/cdf' );
import Uniform = require( './../../../../../base/dists/uniform/ctor' );
import entropy = require( './../../../../../base/dists/uniform/entropy' );
import kurtosis = require( './../../../../../base/dists/uniform/kurtosis' );
import logcdf = require( './../../../../../base/dists/uniform/logcdf' );
import logpdf = require( './../../../../../base/dists/uniform/logpdf' );
import mean = require( './../../../../../base/dists/uniform/mean' );
import median = require( './../../../../../base/dists/uniform/median' );
import mgf = require( './../../../../../base/dists/uniform/mgf' );
import pdf = require( './../../../../../base/dists/uniform/pdf' );
import quantile = require( './../../../../../base/dists/uniform/quantile' );
import skewness = require( './../../../../../base/dists/uniform/skewness' );
import stdev = require( './../../../../../base/dists/uniform/stdev' );
import variance = require( './../../../../../base/dists/uniform/variance' );

/**
* Interface describing the `uniform` namespace.
*/
interface Namespace {
	/**
	* Uniform distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param a - minimum support
	* @param b - maximum support
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 5.0, 0.0, 4.0 );
	* // returns 1.0
	*
	* var mycdf = ns.cdf.factory( 0.0, 10.0 );
	* y = mycdf( 0.5 );
	* // returns 0.05
	*
	* y = mycdf( 8.0 );
	* // returns 0.8
	*/
	cdf: typeof cdf;

	/**
	* Uniform Distribution.
	*/
	Uniform: typeof Uniform;

	/**
	* Returns the differential entropy of a uniform distribution.
	*
	* ## Notes
	*
	* -   If provided `a >= b`, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @returns differential entropy
	*
	* @example
	* var v = ns.entropy( 0.0, 1.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.entropy( 4.0, 12.0 );
	* // returns ~2.079
	*
	* @example
	* var v = ns.entropy( -4.0, 4.0 );
	* // returns ~2.079
	*
	* @example
	* var v = ns.entropy( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( NaN, 2.0 );
	* // returns NaN
	*/
	entropy: typeof entropy;

	/**
	* Returns the excess kurtosis of a uniform distribution.
	*
	* ## Notes
	*
	* -   If provided `a >= b`, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @returns excess kurtosis
	*
	* @example
	* var v = ns.kurtosis( 0.0, 1.0 );
	* // returns -1.2
	*
	* @example
	* var v = ns.kurtosis( 4.0, 12.0 );
	* // returns -1.2
	*
	* @example
	* var v = ns.kurtosis( -4.0, 4.0 );
	* // returns -1.2
	*
	* @example
	* var v = ns.kurtosis( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( NaN, 2.0 );
	* // returns NaN
	*/
	kurtosis: typeof kurtosis;

	/**
	* Uniform distribution logarithm of cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param a - minimum support
	* @param b - maximum support
	* @returns evaluated logCDF
	*
	* @example
	* var y = ns.logcdf( 5.0, 0.0, 4.0 );
	* // returns 0.0
	*
	* var mylogcdf = ns.logcdf.factory( 0.0, 10.0 );
	* y = mylogcdf( 0.5 );
	* // returns ~-2.996
	*
	* y = mylogcdf( 8.0 );
	* // returns ~-0.223
	*/
	logcdf: typeof logcdf;

	/**
	* Uniform distribution natural logarithm of the probability density function (logPDF).
	*
	* @param x - input value
	* @param a - minimum support
	* @param b - maximum support
	* @returns evaluated logPDF
	*
	* @example
	* var y = ns.logpdf( 3.0, 2.0, 6.0 );
	* // returns ~-1.386
	*
	* var mylogpdf = ns.logpdf.factory( 6.0, 7.0 );
	* y = mylogpdf( 7.0 );
	* // returns 0.0
	*
	* y = mylogpdf( 5.0 );
	* // returns -Infinity
	*/
	logpdf: typeof logpdf;

	/**
	* Returns the expected value of a uniform distribution.
	*
	* ## Notes
	*
	* -   If provided `a >= b`, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @returns expected value
	*
	* @example
	* var v = ns.mean( 0.0, 1.0 );
	* // returns 0.5
	*
	* @example
	* var v = ns.mean( 4.0, 12.0 );
	* // returns 8.0
	*
	* @example
	* var v = ns.mean( -4.0, 4.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.mean( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( NaN, 2.0 );
	* // returns NaN
	*/
	mean: typeof mean;

	/**
	* Returns the median of a uniform distribution.
	*
	* ## Notes
	*
	* -   If provided `a >= b`, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @returns median
	*
	* @example
	* var v = ns.median( 0.0, 1.0 );
	* // returns 0.5
	*
	* @example
	* var v = ns.median( 4.0, 12.0 );
	* // returns 8.0
	*
	* @example
	* var v = ns.median( -4.0, 4.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.median( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.median( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.median( NaN, 2.0 );
	* // returns NaN
	*/
	median: typeof median;

	/**
	* Uniform distribution moment-generating function (MGF).
	*
	* @param t - input value
	* @param a - minimum support
	* @param b - maximum support
	* @returns evaluated MGF
	*
	* @example
	* var y = ns.mgf( 2.0, 0.0, 4.0 );
	* // returns ~372.495
	*
	* y = ns.mgf( -0.2, 0.0, 4.0 );
	* // returns ~0.688
	*
	* y = ns.mgf( 2.0, 0.0, 1.0 );
	* // returns ~3.195
	*
	* var mymgf = ns.mgf.factory( 6.0, 7.0 );
	* y = mymgf( 0.1 );
	* // returns ~1.916
	*
	* y = mymgf( 1.1 );
	* // returns ~1339.321
	*/
	mgf: typeof mgf;

	/**
	* Uniform distribution probability density function (PDF).
	*
	* @param x - input value
	* @param a - minimum support
	* @param b - maximum support
	* @returns evaluated PDF
	*
	* @example
	* var y = ns.pdf( 3.0, 2.0, 6.0 );
	* // returns 0.25
	*
	* var myPDF = ns.pdf.factory( 6.0, 7.0 );
	* y = myPDF( 7.0 );
	* // returns 1.0
	*
	* y = myPDF( 5.0 );
	* // returns 0.0
	*/
	pdf: typeof pdf;

	/**
	* Uniform distribution quantile function.
	*
	* @param p - input value
	* @param a - minimum support
	* @param b - maximum support
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.5, 0.0, 10.0 );
	* // returns 5.0
	*
	* y = ns.quantile( 0.8, 0.0, 1.0 );
	* // returns 0.8
	*
	* var myQuantile = ns.quantile.factory( 0.0, 4.0 );
	* y = myQuantile( 0.8 );
	* // returns 3.2
	*/
	quantile: typeof quantile;

	/**
	* Returns the skewness of a uniform distribution.
	*
	* ## Notes
	*
	* -   If provided `a >= b`, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @returns skewness
	*
	* @example
	* var v = ns.skewness( 0.0, 1.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.skewness( 4.0, 12.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.skewness( -4.0, 4.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.skewness( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( NaN, 2.0 );
	* // returns NaN
	*/
	skewness: typeof skewness;

	/**
	* Returns the standard deviation of a uniform distribution.
	*
	* ## Notes
	*
	* -   If provided `a >= b`, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @returns standard deviation
	*
	* @example
	* var v = ns.stdev( 0.0, 1.0 );
	* // returns ~0.289
	*
	* @example
	* var v = ns.stdev( 4.0, 12.0 );
	* // returns ~2.309
	*
	* @example
	* var v = ns.stdev( -4.0, 4.0 );
	* // returns ~2.309
	*
	* @example
	* var v = ns.stdev( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( NaN, 2.0 );
	* // returns NaN
	*/
	stdev: typeof stdev;

	/**
	* Returns the variance of a uniform distribution.
	*
	* ## Notes
	*
	* -   If provided `a >= b`, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @returns variance
	*
	* @example
	* var v = ns.variance( 0.0, 1.0 );
	* // returns ~0.083
	*
	* @example
	* var v = ns.variance( 4.0, 12.0 );
	* // returns ~5.333
	*
	* @example
	* var v = ns.variance( -4.0, 4.0 );
	* // returns ~5.333
	*
	* @example
	* var v = ns.variance( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( NaN, 2.0 );
	* // returns NaN
	*/
	variance: typeof variance;
}

/**
* Uniform distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
