/*
 * This is a shell wrapper to start a user's shell in a change-rooted
 * directory.
 *
 * Note, this program is setuid.  Its action is to change the root
 * directory for the current process and can be invoked by any user.
 * The security of this program is provided by a user/directory
 * mapping for each user to a specified root path.
 */

#define MAXLEN 1024
#define USER_ROOT_PATHS  "/etc/security/changeroot.conf"

#include <stdio.h>
#include <ctype.h>
#include <unistd.h>
/* above should define */
extern int chroot(const char *path);
#include <stdlib.h>
#include <errno.h>
#include <string.h>

#ifdef HAVE_PWDB
#include <pwdb/pwdb_public.h>
#else
#include <pwd.h>
#endif

#ifndef D
#define D(x)
#endif

static void usage(void)
{
    fprintf(stderr, "usage: changeroot command [args..]\n");
    exit(1);
}

int main(int argc, char * const *argv)
{
    FILE *file;
    struct passwd *pw;
    char buffer[MAXLEN], *ptr;

    if (argc < 2) {
	usage();
    }

    /* identify the current user */
    pw = getpwuid(getuid());
    if (pw == NULL) {
	D(("log something?"));
	fprintf(stderr, "changeroot: who are you?\n");
    }

    /* locate their name in the USER_ROOT_PATHS file */

    file = fopen(USER_ROOT_PATHS, "r");
    if (file == NULL) {
	fprintf(stderr, "changeroot: needs a configuration file - %s\n"
		, USER_ROOT_PATHS);
	exit(1);
    }

    while ((ptr = fgets(buffer, MAXLEN, file))) {
	/* skip leading space */
	while (*ptr && isspace(*ptr))
	    ++ptr;

	/* compare with username */
	if (strncmp(pw->pw_name, ptr, strlen(pw->pw_name)) == 0) {
	    /* if match break */
	    break;
	}
    }

    if (file)
	fclose(file);

    if (ptr == NULL) {
	fprintf(stderr, "changeroot: no directory for user\n");
	exit(1);
    }

    /* chroot() to this dir */
    while (*ptr && !isspace(*ptr))
	++ptr;
    while (isspace(*ptr))
	++ptr;

    /* <NUL> terminate */
    {
	char *tmp;

	for (tmp=ptr; *tmp && !isspace(*tmp); ++tmp);
	*tmp = '\0';
    }

    /* chdir() to new '/' */
    if (chroot(ptr) != 0 || chdir("/") != 0) {
	fprintf(stderr, "changeroot: %s\n", strerror(errno));
	exit(1);
    }
    
    /* irrevocably become user (drop setuid bits) */
    setgid(getgid());
    setuid(getuid());

    /* if 0th char of argv[0] is '-', then this is a login shell
       we need to signal this with a similarly modified arg[1] */
    if (argv[0][0] == '-') {
	char ** const args=calloc(argc, sizeof(char *));
	char *vpath=malloc(strlen(argv[1])+2);
	int i;

	if (args == NULL || vpath == NULL) {
	    fprintf(stderr, "changeroot: %s\n", strerror(errno));
	    exit(1);
	}
	vpath[0] = '-';
	strcpy(vpath+1,argv[1]);
	
	args[0] = vpath;
	for (i=1; i<argc; ++i) {
	    fprintf(stderr,"%d:[%s]\n", i,argv[1+i]);
	    args[i] = argv[1+i];
	}

	execv(argv[1], args);
    } else {
	execv(argv[1], 1+argv);
    }

    /* should not get here */
    fprintf(stderr, "changeroot: failed to execute shell\n");
    exit(1);
}
