#ifdef MODVERSIONS
#include <linux/modversions.h>
#endif

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/pagemap.h>
#include <asm/uaccess.h>
#include <asm/fcntl.h>
#include <linux/smp_lock.h>

#include "shfs.h"
#include "shfs_proc.h"

static int shfs_file_readpage(struct file*, struct page*);
static int shfs_file_writepage(struct page*);
static int shfs_file_preparewrite(struct file*, struct page*, unsigned, unsigned);
static int shfs_file_commitwrite(struct file*, struct page*, unsigned, unsigned);
static int shfs_file_permission(struct inode*, int);
static int shfs_file_open(struct inode*, struct file*);
static int shfs_file_flush(struct file*);
static int shfs_file_release(struct inode*, struct file*);
static ssize_t shfs_slow_read(struct file*, char*, size_t, loff_t*);
static ssize_t shfs_slow_write(struct file*, const char*, size_t, loff_t*);

struct file_operations shfs_file_operations = {
	llseek:		generic_file_llseek,
	read:		generic_file_read,
	write:		generic_file_write,
	mmap:		generic_file_mmap,
	open:		shfs_file_open,
	flush:		shfs_file_flush,
	release:	shfs_file_release,
};

struct file_operations shfs_slow_operations = {
	llseek:		generic_file_llseek,
	read:		shfs_slow_read,
	write:		shfs_slow_write,
	mmap:		generic_file_mmap,
	open:		shfs_file_open,
	flush:		shfs_file_flush,
	release:	shfs_file_release,
};

struct inode_operations shfs_file_inode_operations = {
	permission:	shfs_file_permission,
	revalidate:	shfs_revalidate_inode,
	setattr:	shfs_notify_change,
};

struct address_space_operations shfs_file_aops = {
	readpage:	shfs_file_readpage,
	writepage:	shfs_file_writepage,
	sync_page:	block_sync_page,
	prepare_write:	shfs_file_preparewrite,
	commit_write:	shfs_file_commitwrite
};

static int
shfs_file_readpage(struct file *f, struct page *p)
{
	struct dentry *dentry = f->f_dentry;
	struct shfs_sb_info *info = (struct shfs_sb_info*)dentry->d_sb->u.generic_sbp;
	char *buffer;
	unsigned long offset, count;
	int result;
	
	get_page(p);

	buffer = kmap(p);
	offset = p->index << PAGE_CACHE_SHIFT;
	count = PAGE_SIZE;

	lock_kernel();
	do {
		DEBUG(" \n");
		if (!shfs_lock(info)) {
			VERBOSE("Interrupted\n");
			unlock_kernel();
			result = -EINTR;
			goto io_error;
			
		}
		if (!info->mnt.disable_fcache)
			result = shfs_fcache_read(f, offset, count, buffer);
		else
			result = shfs_proc_read(dentry, offset, count, buffer, 0);
		shfs_unlock(info);
		if (result < 0) {
			VERBOSE(" IO error! (%d)\n", result);
			unlock_kernel();
			goto io_error;
		}
		count -= result;
		offset += result;
		buffer += result;
		dentry->d_inode->i_atime = CURRENT_TIME;
		if (!result)
			break;
	} while (count);

	unlock_kernel();
	memset(buffer, 0, count);
	flush_dcache_page(p);
	SetPageUptodate(p);
	result = 0;
io_error:
	kunmap(p);
	UnlockPage(p);	
	put_page(p);
	return shfs_remove_sigpipe(result);
}

static int
shfs_file_writepage(struct page *p)
{
	VERBOSE(" This should't happen!!\n");
	return -EFAULT;
}

static int
shfs_file_preparewrite(struct file *f, struct page *p, unsigned offset, unsigned to)
{
	DEBUG("page: %p, offset: %u, to: %u\n", p, offset, to);
//	kmap(p);
	return 0;
}

static int
shfs_file_commitwrite(struct file *f, struct page *p, unsigned offset, unsigned to)
{
	struct dentry *dentry = f->f_dentry;
	struct shfs_sb_info *info = (struct shfs_sb_info*)dentry->d_sb->u.generic_sbp;
	struct inode *inode = p->mapping->host;
	char *buffer = kmap(p) + offset;
	struct iattr attr;
	int written = 0, result;
	unsigned count = to - offset;
	
	DEBUG("page: %p, offset: %u, to: %u\n", p, offset, to);
	offset += p->index << PAGE_CACHE_SHIFT;

	lock_kernel();
	
	if (info->mnt.readonly) {
		result = -EROFS;
		goto error;
	}

	do {
		if (!shfs_lock(info)) {
			VERBOSE("Interrupted\n");
			result = -EINTR;
			goto error;
		}
		if (!info->mnt.disable_fcache)
			result = shfs_fcache_write(f, offset, count, buffer);
		else
			result = shfs_proc_write(dentry, offset, count, buffer);
		shfs_unlock(info);
		if (result < 0) {
			VERBOSE(" IO error! (%d)\n", result);
			goto error;
		}

		count -= result;
		offset += result;
		buffer += result;
		written += result;
		if (!result)
			break;

	} while (count);

	memset(buffer, 0, count);
	result = 0;
error:
	unlock_kernel();
	kunmap(p);

	DEBUG("offset: %u, i_size: %lld\n", offset, inode->i_size);
	inode->i_mtime = inode->i_atime = CURRENT_TIME;
	attr.ia_valid = ATTR_ATIME;
	attr.ia_atime = CURRENT_TIME;
	if (offset > inode->i_size) {
		inode->i_size = offset;
		attr.ia_valid |= ATTR_SIZE;
		attr.ia_size = offset;
	}
	if (!shfs_lock(info)) {
		VERBOSE("Interrupted\n");
		return -EINTR;
	}
	shfs_dcache_update_fattr(dentry, &attr);
	shfs_unlock(info);
	return shfs_remove_sigpipe(result);
}

static int
shfs_file_permission(struct inode *inode, int mask)
{
	int mode = inode->i_mode;

	mode >>= 6;
	if ((mode & 7 & mask) != mask)
		return -EACCES;
	return 0;
}

static int
shfs_file_open(struct inode *inode, struct file *f)
{
	struct dentry *dentry = f->f_dentry;
	struct shfs_sb_info *info = (struct shfs_sb_info*)dentry->d_sb->u.generic_sbp;
	int mode = f->f_flags & O_ACCMODE;
	char buf[SHFS_PATH_MAX];
	int error = 0;

	DEBUG(" %s\n", dentry->d_name.name);

	if ((mode == O_WRONLY || mode == O_RDWR) && info->mnt.readonly)
		return -EROFS;

	if (shfs_get_name(dentry, buf) < 0) {
		VERBOSE("Name too long!\n");
		return -ENAMETOOLONG;
	}

	error = shfs_proc_open(info, buf, mode);

	if (!shfs_lock(info)) {
		VERBOSE("Interrupted\n");
		return -EINTR;
	}
	switch (error) {
	case 1:			/* install special read handler for this file */
		if (dentry->d_inode)
			f->f_op = &shfs_slow_operations;
		error = 0;	/* fall through */
	case 0:
		shfs_fcache_add(f);
		break;
	default:		/* error */
		DEBUG("open error: %d\n", error);
		break;
	}
	shfs_unlock(info);

	return shfs_remove_sigpipe(error);
}

static int
shfs_file_flush(struct file *f)
{
	struct dentry *dentry = f->f_dentry;
	struct shfs_sb_info *info = (struct shfs_sb_info*)dentry->d_sb->u.generic_sbp;
	int res;

	DEBUG(" %s\n", dentry->d_name.name);
	res = 0;
	if (!info->mnt.disable_fcache) {
		if (!shfs_lock(info)) {
			VERBOSE("Interrupted\n");
			return -EINTR;
		}
		res = shfs_fcache_sync(f);
		shfs_unlock(info);
	}
	return shfs_remove_sigpipe(res < 0 ? res : 0);
}

static int
shfs_file_release(struct inode *inode, struct file *f)
{
	struct dentry *dentry = f->f_dentry;
	struct shfs_sb_info *info = (struct shfs_sb_info*)dentry->d_sb->u.generic_sbp;
	int error = 0, last;
	
	DEBUG(" %s\n", dentry->d_name.name);
	if (!info->mnt.disable_fcache) {
		shfs_locku(info);		/* no way to return error */
		error = shfs_fcache_remove(f, &last);
		if (error < 0)
			shfs_dcache_invalidate_dir(dentry->d_parent->d_inode);
		shfs_unlock(info);
		if (last) {
			/* last reference */
			lock_kernel();
			filemap_fdatasync(dentry->d_inode->i_mapping);
			filemap_fdatawait(dentry->d_inode->i_mapping);
			unlock_kernel();
		}
	}
	return shfs_remove_sigpipe(error);
}

static ssize_t 
shfs_slow_read(struct file *f, char *buf, size_t count, loff_t *ppos)
{
	struct shfs_sb_info *info = (struct shfs_sb_info*)f->f_dentry->d_sb->u.generic_sbp;
	unsigned long page;
	int error = 0;
	
	DEBUG("%s\n", f->f_dentry->d_name.name);

	lock_kernel();
	if (!shfs_lock(info)) {
		VERBOSE("Interrupted\n");
		unlock_kernel();
		return -EINTR;
	}
	page = __get_free_page(GFP_KERNEL);
	if (!page) {
		error = -ENOMEM;
		goto out;
	}

	count = count > PAGE_SIZE ? PAGE_SIZE : count;
	error = shfs_proc_read(f->f_dentry, *ppos, count, (char *)page, 1);
	if (error < 0) {
		VERBOSE(" IO error! (%d)\n", error);
		unlock_kernel();
		goto error;
	}
	if (error != 0) {
		copy_to_user(buf, (char *)page, error);
		*ppos += error;
	}
error:
	free_page(page);
out:
	shfs_unlock(info);
	unlock_kernel();
	return shfs_remove_sigpipe(error);
}

static ssize_t 
shfs_slow_write(struct file *f, const char *buf, size_t count, loff_t *offset)
{
	int written = 0;
	int error;
	
	DEBUG("\n");
	written = generic_file_write(f, buf, count, offset);
	if (written > 0) {
		error = shfs_file_flush(f);
		written = error < 0 ? error: written;
	}
	
	return shfs_remove_sigpipe(written);
}

